package com.burhanstore.neonbubblequest.Utils.monetize

import android.app.Activity
import android.util.Log
import com.burhanstore.neonbubblequest.Utils.DialogManager.dismissDialog
import com.google.android.gms.ads.AdError
import com.google.android.gms.ads.FullScreenContentCallback
import com.google.android.gms.ads.LoadAdError
import com.google.android.gms.ads.MobileAds
import com.google.android.gms.ads.admanager.AdManagerAdRequest
import com.google.android.gms.ads.rewarded.RewardItem
import com.google.android.gms.ads.rewarded.RewardedAd
import com.google.android.gms.ads.rewarded.RewardedAdLoadCallback
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

object Admob_RewardAd {
    private var rewardedAd: RewardedAd? = null
    private var isAdLoading = false
    private var isAdLoaded = false

    private val TAG = "AdxAdapter"

    fun initializeSdk(activity: Activity) {
        CoroutineScope(Dispatchers.IO).launch {
            MobileAds.initialize(activity) {
                Log.d(TAG, "MobileAds initialized")
            }
        }
    }

    fun loadAd(activity: Activity) {
        if (isAdLoading) {
            Log.d(TAG, "Ad is already loading")
            return
        }

        isAdLoading = true
        val adRequest = AdManagerAdRequest.Builder().build()
        val adUnitId = "ca-app-pub-3940256099942544/5224354917"
        RewardedAd.load(
            activity,
            adUnitId.toString(),
            adRequest,
            object : RewardedAdLoadCallback() {
                override fun onAdLoaded(ad: RewardedAd) {
                    Log.d(TAG, "Ad loaded successfully")
                    rewardedAd = ad
                    isAdLoading = false
                    isAdLoaded = true
                }

                override fun onAdFailedToLoad(error: LoadAdError) {
                    Log.e(TAG, "Failed to load ad: ${error.message}")
                    rewardedAd = null
                    isAdLoading = false
                    isAdLoaded = false
                }
            })

        Log.d(TAG, "Ad is already loading"+ adUnitId)
    }

    fun showAd(
        activity: Activity,
        onUserEarnedReward: (RewardItem) -> Unit,
        onAdComplete: () -> Unit = {},
        onAdFailed: () -> Unit = {},
        onAdClicked: () -> Unit = {},
    ) {
        if (rewardedAd != null && isAdLoaded) {
            var hasEarnedReward = false

            rewardedAd?.fullScreenContentCallback = object : FullScreenContentCallback() {
                override fun onAdClicked() {
                    Log.d(TAG, "Ad clicked")
                    onAdClicked()
                }

                override fun onAdDismissedFullScreenContent() {
                    Log.d(TAG, "Ad dismissed")
                    rewardedAd = null
                    isAdLoaded = false
                    loadAd(activity)

                    if (!hasEarnedReward) {
                        Log.d(TAG, "User didn't finish the ad, no reward given")
                    }

                    onAdComplete()
                }

                override fun onAdFailedToShowFullScreenContent(adError: AdError) {
                    Log.e(TAG, "Failed to show ad: ${adError.message}")
                    rewardedAd = null
                    isAdLoaded = false
                    onAdFailed()
                }

                override fun onAdImpression() {
                    dismissDialog()
                    Log.d(TAG, "Ad impression recorded")
                }

                override fun onAdShowedFullScreenContent() {
                    Log.d(TAG, "Ad is showing full screen content")
                }
            }

            rewardedAd?.show(activity) { rewardItem ->
                hasEarnedReward = true
                Log.d(TAG, "User earned reward: ${rewardItem.amount} ${rewardItem.type}")
                onUserEarnedReward(rewardItem)
            }
        } else {
            Log.w(TAG, "Ad is not ready yet")
            onAdFailed()
            loadAd(activity)
        }
    }


    fun isAdReady(): Boolean {
        return rewardedAd != null && isAdLoaded
    }

    fun destroy() {
        rewardedAd = null
        isAdLoaded = false
        isAdLoading = false
    }

}